package main

import (
	"bytes"
	"crypto/hmac"
	"crypto/sha256"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strings"
	"time"
)

var itsAppid = "xxxx"
var itsKey = "xxxx"
var itsSecret = "xxxx"

// 请求地址
var itsRequestUrl = "https://itrans.xf-yun.com/v1/its"

type itsRequestData struct {
	Header struct {
		Appid  string `json:"app_id"`
		Status int    `json:"status"`
		ResId  string `json:"res_id"`
	} `json:"header"`
	Parameter struct {
		Its struct {
			From   string   `json:"from"`
			To     string   `json:"to"`
			Result struct{} `json:"result"`
		} `json:"its"`
	} `json:"parameter"`
	Payload struct {
		InputData struct {
			Encoding string `json:"encoding"`
			Status   int    `json:"status"`
			Text     string `json:"text"`
		} `json:"input_data"`
	} `json:"payload"`
}

// 使用匿名结构体解析 JSON
type itsResponseData struct {
	Header struct {
		Code    int    `json:"code"`
		Message string `json:"message"`
		Sid     string `json:"sid"`
	} `json:"header"`
	Payload struct {
		Result struct {
			Req    string `json:"req"`
			Text   string `json:"text"`
			Status string `json:"status"`
		} `json:"result"`
	} `json:"payload"`
}

func main() {
	var inputParam itsRequestData
	inputParam.Header.Appid = itsAppid
	inputParam.Header.Status = 3
	inputParam.Parameter.Its.From = "cn"
	inputParam.Parameter.Its.To = "en"
	inputParam.Payload.InputData.Encoding = "utf8"
	inputParam.Payload.InputData.Status = 3

	encodedData := base64.StdEncoding.EncodeToString([]byte("我是中国人，我来自安徽省合肥市"))
	inputParam.Payload.InputData.Text = encodedData
	requestJSON, err := json.Marshal(inputParam)
	if err != nil {
		fmt.Println("json.Marshal err: ", err)
		return
	}
	autUrl := assembleAuthUrlForPost(itsRequestUrl, itsKey, itsSecret)
	parsedURL, err := url.Parse(autUrl)
	if err != nil {
		fmt.Println("parse url err: ", err)
		return
	}
	req, err := http.NewRequest("POST", autUrl, bytes.NewBuffer(requestJSON))
	if err != nil {
		fmt.Println("NewRequest err: ", err)
		return
	}
	req.Header.Set("content-type", "application/json")
	req.Header.Set("host", parsedURL.Hostname())
	req.Header.Set("app_id", itsAppid)

	client := &http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		fmt.Println("Do err: ", err)
		return
	}
	defer resp.Body.Close()

	var respData itsResponseData
	err = json.NewDecoder(resp.Body).Decode(&respData)
	if err != nil {
		fmt.Println("JSON decode err: ", err)
		return
	}
	if resp.StatusCode != 200 {
		fmt.Printf("respData err code: %v; detail: %v\n", respData.Header.Code, respData.Header.Message)
		return
	}
	decodedData, err := base64.StdEncoding.DecodeString(respData.Payload.Result.Text)
	if err != nil {
		fmt.Println("base64 decode err: ", err)
		return
	}
	fmt.Println("translate result: ", string(decodedData))
}

func assembleAuthUrlForPost(hostUrl string, apiKey, apiSecret string) string {
	ul, err := url.Parse(hostUrl)
	if err != nil {
		fmt.Println("parse url err: ", err)
		return ""
	}

	date := time.Now().UTC().Format("Mon, 02 Jan 2006 15:04:05 GMT")
	signString := []string{
		"host: " + ul.Hostname(),
		"date: " + date,
		"POST " + ul.Path + " HTTP/1.1",
	}

	sgin := strings.Join(signString, "\n")
	sha := hmacWithShaToBase64("hmac-sha256", sgin, apiSecret)
	authUrl := fmt.Sprintf("api_key=\"%s\", algorithm=\"%s\", headers=\"%s\", signature=\"%s\"", apiKey,
		"hmac-sha256", "host date request-line", sha)

	authorization := base64.StdEncoding.EncodeToString([]byte(authUrl))

	v := url.Values{}
	v.Add("host", ul.Hostname()) // 使用 Hostname() 而不是 Host
	v.Add("date", date)
	v.Add("authorization", authorization)

	callUrl := hostUrl + "?" + v.Encode()
	return callUrl
}

func hmacWithShaToBase64(algorithm, data, key string) string {
	mac := hmac.New(sha256.New, []byte(key))
	mac.Write([]byte(data))
	encodeData := mac.Sum(nil)
	return base64.StdEncoding.EncodeToString(encodeData)
}
