package main

import (
	"bytes"
	"crypto/hmac"
	"crypto/sha256"
	"encoding/base64"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"os"
	"strings"
	"time"
)

var ocrAppid = "这里填写控制台的APPID"
var ocrKey = "这里填写控制台的APIKEY"
var ocrSecret = "这里填写控制台的APISECRECT"

// 请求地址
var ocrRequestUrl = "https://cn-huadong-1.xf-yun.com/v1/inv"

// 请求数据结构
type RequestData struct {
	Header    map[string]interface{}            `json:"header"`
	Parameter map[string]map[string]interface{} `json:"parameter"`
	Payload   map[string]map[string]interface{} `json:"payload"`
}

// 响应数据结构
// 使用匿名结构体解析 JSON
type responseData struct {
	Header struct {
		Code    int    `json:"code"`
		Message string `json:"message"`
		Sid     string `json:"sid"`
		Status  string `json:"status"`
	} `json:"header"`
	Payload struct {
		Result struct {
			Encoding string `json:"encoding"`
			Compress string `json:"compress"`
			Format   string `json:"format"`
			Text     string `json:"text"`
			Status   string `json:"status"`
		} `json:"result"`
	} `json:"payload"`
}

var requestData = RequestData{
	Header: map[string]interface{}{
		"app_id": ocrAppid,
		"status": 3,
	},
	Parameter: map[string]map[string]interface{}{
		"ocr": {
			"type":  "air_itinerary",
			"level": 1,
			"result": map[string]interface{}{
				"encoding": "utf8",
				"compress": "raw",
				"format":   "json",
			},
		},
	},
	Payload: map[string]map[string]interface{}{
		"image": {
			"encoding": "jpg",
			"image":    "",
			"status":   3,
		},
	},
}

// 获取文件字节数据
func getFileBytes(filePath string) ([]byte, error) {
	if _, err := os.Stat(filePath); os.IsNotExist(err) {
		return nil, errors.New("file does not exist")
	}

	file, err := os.Open(filePath)
	if err != nil {
		return nil, err
	}
	defer file.Close()

	// 检查是否为WAV文件并跳过头部
	buffer := make([]byte, 4)
	_, err = file.Read(buffer)
	if err != nil {
		return nil, err
	}

	// 如果是WAV文件，跳过44字节头部
	var startPos int64
	if bytes.Equal(buffer, []byte("RIFF")) {
		startPos = 44
	} else {
		startPos = 0
	}

	_, err = file.Seek(startPos, 0)
	if err != nil {
		return nil, err
	}

	return io.ReadAll(file)
}
func main() {
	imgData, err := getFileBytes("ticket_img.png")
	if err != nil {
		fmt.Println("getFileBytes err: ", err)
		return
	}
	requestData.Payload["image"]["image"] = base64.StdEncoding.EncodeToString(imgData)
	requestJSON, err := json.Marshal(requestData)
	if err != nil {
		fmt.Println("json.Marshal err: ", err)
		return
	}
	autUrl := assembleAuthUrlForPost(ocrRequestUrl, ocrKey, ocrSecret)
	parsedURL, err := url.Parse(autUrl)
	if err != nil {
		fmt.Println("parse url err: ", err)
		return
	}
	req, err := http.NewRequest("POST", autUrl, bytes.NewBuffer(requestJSON))
	if err != nil {
		fmt.Println("NewRequest err: ", err)
		return
	}
	req.Header.Set("content-type", "application/json")
	req.Header.Set("host", parsedURL.Hostname())
	req.Header.Set("app_id", ocrAppid)

	client := &http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		fmt.Println("Do err: ", err)
		return
	}
	defer resp.Body.Close()
	var respData responseData
	err = json.NewDecoder(resp.Body).Decode(&respData)
	if err != nil {
		fmt.Println("JSON decode err: ", err)
		return
	}
	decodedData, err := base64.StdEncoding.DecodeString(respData.Payload.Result.Text)
	if err != nil {
		fmt.Println("base64 decode err: ", err)
		return
	}

	err = ioutil.WriteFile("output1.bin", decodedData, 0644)
	if err != nil {
		fmt.Println("writeFile err:", err)
		return
	}
	fmt.Println("write success!")
}

func assembleAuthUrlForPost(hostUrl string, apiKey, apiSecret string) string {
	ul, err := url.Parse(hostUrl)
	if err != nil {
		fmt.Println("parse url err: ", err)
		return ""
	}

	date := time.Now().UTC().Format("Mon, 02 Jan 2006 15:04:05 GMT")
	signString := []string{
		"host: " + ul.Hostname(),
		"date: " + date,
		"POST " + ul.Path + " HTTP/1.1",
	}

	sgin := strings.Join(signString, "\n")
	sha := hmacWithShaToBase64("hmac-sha256", sgin, apiSecret)
	authUrl := fmt.Sprintf("api_key=\"%s\", algorithm=\"%s\", headers=\"%s\", signature=\"%s\"", apiKey,
		"hmac-sha256", "host date request-line", sha)

	authorization := base64.StdEncoding.EncodeToString([]byte(authUrl))

	v := url.Values{}
	v.Add("host", ul.Hostname()) // 使用 Hostname() 而不是 Host
	v.Add("date", date)
	v.Add("authorization", authorization)

	callUrl := hostUrl + "?" + v.Encode()
	return callUrl
}

// 修复的 HMAC 函数
func hmacWithShaToBase64(algorithm, data, key string) string {
	mac := hmac.New(sha256.New, []byte(key))
	mac.Write([]byte(data))
	return base64.StdEncoding.EncodeToString(mac.Sum(nil))
}
